<?php
class GalaxPayCreditCard extends GalaxPayBaseGateway
{
    private $max_installments = 12;

    public function __construct(GalaxPaySettings $container)
    {
        $this->id = 'galax-pay-credit-card';
        $this->support = array('products', 'subscriptions');
        $this->method_description = 'Pagamentos via cartão de crédito processados pelo Galax Pay.';
        $this->method_title = __('Galax Pay - Cartão de Crédito', GALAX_IDENTIFIER);
        $this->has_fields = true;
        $this->init_form_fields();
        $this->init_settings();
        $this->smallest_installment = $this->get_option('smallest_installment');
        $this->installments = $this->get_option('installments');
        $this->verify_method = $this->get_option('verify_method');
        $this->supports  = GalaxPaySettings::getSupports();
        parent::__construct($container);
        add_action('wp_enqueue_scripts', array(&$this, 'checkout_script'));
    }

    public function validate_installments_table_field( $key, $value ) {
		return $value;
	}

    public function init_form_fields()
    {
        $this->form_fields = array(
            'enabled' => array(
                'title' => __('Habilitar/Desabilitar', GALAX_IDENTIFIER),
                'label' => __('Habilitar pagamento via Cartão de Crédito com o Galax Pay', GALAX_IDENTIFIER),
                'type' => 'checkbox',
                'default' => 'no',

            ),
            'title' => array(
                'title' => __('Título', GALAX_IDENTIFIER),
                'type' => 'text',
                'description' => __('Título que o cliente verá durante o processo de pagamento.', GALAX_IDENTIFIER),
                'default' => __('Cartão de Crédito', GALAX_IDENTIFIER),
            ),
            'single_charge' => array(
                'title' => __('Vendas Avulsas', GALAX_IDENTIFIER),
                'type' => 'title',
            ),
            'smallest_installment' => array(
                'title' => __('Valor mínimo da parcela', GALAX_IDENTIFIER),
                'type' => 'text',
                'description' => __('Valor mínimo da parcela, não deve ser inferior a R$ 5,00.', GALAX_IDENTIFIER),
                'default' => '5',

            ),
            'installments' => array(
                'title' => __('Número máximo de parcelas', GALAX_IDENTIFIER),
                'type' => 'select',
                'description' => __('Número máximo de parcelas para vendas avulsas. Deixe em 1x para desativar o parcelamento.', GALAX_IDENTIFIER),
                'default' => '1',
                'options' => array(
                    '1' => '1x',
                    '2' => '2x',
                    '3' => '3x',
                    '4' => '4x',
                    '5' => '5x',
                    '6' => '6x',
                    '7' => '7x',
                    '8' => '8x',
                    '9' => '9x',
                    '10' => '10x',
                    '11' => '11x',
                    '12' => '12x',
                ),
            ),
            'installments_fee_header' => array(
                'title'        => __( 'Parcelamento', 'woo-juno' ),
                'type'         => 'title',
                'description'  => '<div id="installments_fee_header">' . __( 'Por padrão, o parcelamento é exibido sem juros. Na tabela abaixo você pode definir os juros em porcentagem por parcela. Cada parcela terá os juros totais que você irá repassar (ex.: se você quiser adicionar 1% por parcela acima de 3x, irá adicionar 4% para 4x, 5% para 5x, 6% para 6x e assim por diante). Caso as outras parcelas estejam em branco, serão consideradas sem juros.', 'woo-juno' ) . '</div>'
              ),
            'installments_fee' => array(
                'type' => 'installments_table',
                'title' => __( 'Parcelas', 'woo-juno' ),
              ),

        );
    }

    public function generate_installments_table_html( $key, $data ) {
		$field_key = $this->get_field_key( $key );
		$value     = $this->get_option( $key );
		$defaults  = array(
			'title'             => '',
			'disabled'          => false,
			'class'             => '',
			'css'               => '',
			'placeholder'       => '',
			'type'              => 'text',
			'desc_tip'          => false,
			'description'       => '',
			'custom_attributes' => array(),
		);

		$data = wp_parse_args( $data, $defaults );

		ob_start();
		?>
		<tr valign="top">
			<th scope="row" class="titledesc">
				<label for="<?php echo esc_attr( $field_key ); ?>"><?php echo wp_kses_post( $data['title'] ); ?> <?php echo $this->get_tooltip_html( $data ); // WPCS: XSS ok. ?></label>
			</th>
			<td class="forminp">
        <fieldset>
          <?php
            for ( $i = 1; $i <= 12; $i++ ) :
              $interest = isset( $value[ $i ] ) ? $value[ $i ] : '';
          ?>
          <p data-installment="<?php echo $i; ?>">
            <input class="small-input" type="text" value="<?php echo $i; ?>"
              <?php disabled( 1, true ); ?> />
            <input class="small-input wc_input_price" type="text"
              placeholder="0,00 (valor em porcentagem, somente números)"
              name="<?php echo esc_attr( $field_key ); ?>[<?php echo $i; ?>]"
              id="<?php echo esc_attr( $field_key ); ?>_<?php echo $i; ?>" value="<?php echo wc_format_localized_price( $interest ) ?>" />
          </p>
          <?php endfor; ?>
        </fieldset>
			</td>
		</tr>
		<?php

		return ob_get_clean();
	}


    public function is_available()
    {
        if (false === is_checkout()) {
            return false;
        }
        return $this->enabled && $this->container->check_ssl() == 'yes';
    }

    public function verify_method()
    {
        return $this->verify_method == 'yes';
    }

    public function payment_fields()
    {
        $total = $this->container->woocommerce->cart->total;
        $max_times = $this->get_order_max_installments($total);
        if ($max_times > 1) {

            // for ($times = 1; $times <= $max_times; $times++) {

            //     $installments[$times] = ceil($total / $times * 100) / 100;
            // }
            $installments = $this->get_installments_options();
        } else {

            $installments = 1;
        }
        
        if (GalaxPayGetInfoOrder::hasSubscriptionAdditionalPlugin()) {
            $installments = 1; //assinatura

        }
        $this->container->get_template('creditcard-checkout.html.php', compact('installments'));
    }


	protected function get_installments_options () {
        $total = $this->get_order_total();
		$max_installments = $this->get_order_max_installments($total);
        

		if ( $this->validate_installments( $max_installments ) && $max_installments > 1 ) {
			
			$options = [];
			$label_singular = __( '%s parcela de %s %s', 'woo-juno' );
			$label_plural = __( '%s parcelas de %s %s', 'woo-juno' );

			foreach ( \range(1, $max_installments ) as $i ) {
				$fees   = $this->get_option( 'installments_fee' );
				$fee    = ! empty( $fees[ $i ] ) ? wc_format_decimal( $fees[ $i ] ) : 0;

				$amount     = $total / $i;
				$final_cost = $this->get_final_cost( $amount, $i );
				$options[ $i ] = \wptexturize( \sprintf(
					$i == 1 ? $label_singular : $label_plural,
					$i,
					\strip_tags( \wc_price( $final_cost ) ),
					0 === $fee ? 'sem juros' : ''
				) );
			}

			return apply_filters( $this->get_option( 'installments_list' ), $options, $this );
		}

		return false;
	}

    protected function validate_installments ( $value ) {
        
        $total = $this->get_order_total();
		$max_installments = $this->get_order_max_installments($total);
		$value = intval( $value );

		if ( $max_installments < 0 ) {
			$max_installments = 1;
		} elseif ( $max_installments > 12 ) {
			$max_installments = 12;
		}

		if ( $value < 1 || $value > $max_installments ) {
			return false;
		}

		return true;
	}

	public function get_final_cost( $total, $installments ) {
		$fees   = $this->get_option( 'installments_fee' );
		$fee    = isset( $fees[ $installments ] ) ? wc_format_decimal( $fees[ $installments ] ) : 0;

		if ( $fee > 0 ) {
			$total += $total * ( $fee / 100 );
		}

		return wc_format_decimal( $total, 2, true );
	}

    protected function get_installments()
    {
        return 12;
    }

    protected function get_order_max_installments($order_total)
    {
        if ($this->is_single_order()) {
            $this->max_installments = $this->installments;
            $order_max_times = floor($order_total / $this->smallest_installment);
            $max_times = empty($order_max_times) ? 1 : $order_max_times;
            return min($this->max_installments, $max_times, $this->get_installments());
        }
        return 1;
    }

    public function validate_fields()
    {
        if ($this->is_single_order() && $this->installments < 1) {
            $installments = sanitize_text_field($_POST['installmentsNumber']);
            if (!isset($installments) || empty($installments)) {
                wc_add_notice('Quantidade de parcelas inválida.', 'error');
            }
            $total = $this->container->woocommerce->cart->total;
            if ($installments > $this->get_order_max_installments($total)) {
                wc_add_notice('A quantidade de parcelas escolhidas é inválida.', 'error');
            }
        }
        $fields = array(
            'cardHolder' => 'nome impresso no cartão',
            'cardNumber' => 'número do cartão',
            'cardCvv' => 'código de segurança',
            'cardMonthExpiry' => 'mês de validade',
            'cardYearExpiry' => 'ano de validade ',
        );

        foreach ($fields as $field => $name) {
            $postField = $_POST[$field];
            if (!isset($postField) || empty($postField)) {
                $message = 'O campo "<b>' . $name . '</b>" é obrigatório!';
                wc_add_notice($message, 'error');
                throw new \Exception();
            }
        }
        $now = time();
        $ccExpiry = mktime(0, 0, 0, (int) $this->getCardMonthExpiry(), 1, (int) $this->getCardYearExpiry());
        if ($now >= $ccExpiry) {
            if (date('m') == $this->getCardMonthExpiry()) {
                return true;
            }
            wc_add_notice('Cartão expirado.', 'error');
            throw new \Exception();
        }
    }

    public function getCardMonthExpiry()
    {
        $cardMonthExpiry = sanitize_text_field($_POST['cardMonthExpiry']);
        if ($cardMonthExpiry) {
            return $cardMonthExpiry;
        }
    }

    public function getCardYearExpiry()
    {
        $cardYearExpiry = sanitize_text_field($_POST['cardYearExpiry']);
        if ($cardYearExpiry) {
            return $cardYearExpiry;
        }
    }

    public function checkout_script()
    {
        if (!(get_query_var('order-received')) && is_checkout()) {
            $this->container->add_script('js/checkout.js', array('jquery', 'jquery-payment'));
        }
    }

    public function admin_options() {
		parent::admin_options();

		?>
		<script type="text/javascript">
			jQuery(document).ready(function($) {
				$( document.body ).on( 'change', '#galax-pay-woocommerce-credit-card_integration_method', function( event ) {
					if ( 'redirect' === $(this).val() ) {
						$( '#galax-pay-woocommerce-credit-card_show_visual_card, #galax-pay-woocommerce-credit-card_store_user_cards, #galax-pay-woocommerce-credit-card_smallest_installment, #galax-pay-woocommerce-credit-card_payment_advance' ).closest( 'tr' ).hide();

						$( '#galax-pay-woocommerce-credit-card_installments_fee_header, #installments_fee_header' ).hide();

						$( '#galax-pay-woocommerce-credit-card_installments_fee_1' ).closest( 'table' ).hide();
					} else {
						$( '#galax-pay-woocommerce-credit-card_show_visual_card, #galax-pay-woocommerce-credit-card_store_user_cards, #galax-pay-woocommerce-credit-card_smallest_installment, #galax-pay-woocommerce-credit-card_payment_advance' ).closest( 'tr' ).show();

						$( '#galax-pay-woocommerce-credit-card_installments_fee_header, #installments_fee_header' ).show();

						$( '#galax-pay-woocommerce-credit-card_installments_fee_1' ).closest( 'table' ).show();
					}
				});

				$( '#galax-pay-woocommerce-credit-card_integration_method' ).change();
			});
		</script>
		<?php
	}
}
