<?php
class GalaxPayBoleto extends GalaxPayBaseGateway
{

    public function __construct(GalaxPaySettings $container)
    {
        $this->id = 'galax-pay-boleto';
        $this->method_description = 'Pagamentos via boleto bancário processados pelo Galax Pay.';
        $this->method_title = __('Galax Pay - Boleto Bancário', GALAX_IDENTIFIER);
        $this->has_fields = true;
        $this->init_form_fields();
        $this->init_settings();
        add_action('woocommerce_thankyou_' . $this->id, array(&$this, 'thank_you_page'));
        add_filter('woocommerce_my_account_my_orders_actions', array(&$this, 'my_account_print_buttons'), 10, 2);
        $this->supports  = GalaxPaySettings::getSupports();
        parent::__construct($container);
    }

    public function is_available()
    {
        if (is_checkout() == false) {
            return false;
        }
        return $this->enabled && $this->container->api->accept_bank_slip() && $this->container->check_ssl() == 'yes';
    }

    public function payment_fields()
    {
        $this->container->get_template('boleto-checkout.html.php');
    }

    public function thank_you_page($order_id)
    {
        $response = $this->container->api->getChargeByMyId($order_id);
        $name = 'Charges';
        if ($response['totalQtdFoundInPage'] == 0) {
            $response = $this->container->api->getSubscriptionById($order_id);
            $name = 'Subscriptions';
        }

        $transaction = $response[$name]['0']['Transactions']['0'];
        if (!empty($transaction['Boleto']['pdf'])) {
            $boletoUrl = $transaction['Boleto']['pdf'];
            $this->container->get_template('boleto-download.html.php', compact('boletoUrl'));
        }
    }

    public function init_form_fields()
    {
        $this->form_fields = array(
            'enabled' => array(
                'title' => __('Habilitar/Desabilitar', GALAX_IDENTIFIER),
                'label' => __('Habilitar pagamento por Boleto Bancário com o Galax Pay', GALAX_IDENTIFIER),
                'type' => 'checkbox',
                'default' => 'no',
            ),
            'title' => array(
                'title' => __('Título', GALAX_IDENTIFIER),
                'type' => 'text',
                'description' => __('Título que o cliente verá durante o processo de pagamento.', GALAX_IDENTIFIER),
                'default' => __('Boleto Bancário', GALAX_IDENTIFIER),
            ),
            

            'saleConfig' => array(

                'title' => __('Configurações do boleto', GALAX_IDENTIFIER),

                'type' => 'title',

            ),

            'daysAfterBoletoDueDate' => array(

                'title' => __('Prazo máximo para pagamento após o vencimento', GALAX_IDENTIFIER),

                'type' => 'select',

                'description' => '',

                'default' => '1',

                'options' => array(

                    '0' => 'Dia atual',

                    '1' => 'Um dia após a compra',

                    '2' => 'Dois dias após a compra',

                    '3' => 'Três dias após a compra',

                    '4' => 'Quatro dias após a compra',

                    '5' => 'Cinco dias após a compra',

                    '6' => 'Seis dias após a compra',

                    '7' => 'Sete dias após a compra',

                )

            ),

            'infoBoleto' => array(

                'title' => __('Informações do Boleto'),

                'type' => 'textarea',

                'description' => __('Informações adicionais para serem impressas no boleto. Podem ser instruções de pagamento ou descrição do serviço prestado. Máximo de 255 caracteres.'),

                'default' => 'Pagamento feito através de loja virtual'

            ),
        );
    }

    public function my_account_print_buttons($actions, $order)
    {
        if ($this->id !== $order->get_payment_method()) {
            return $actions;
        }
        $response = $this->container->api->getChargeByMyId($order->id);
        if (!isset($response['Charges'])) {
            return;
        }
        if (!isset($response['Charges']['0'])) {
            return;
        }
        $transaction = $response['Charges']['0']['Transactions']['0'];
        $link = $transaction['Boleto']['pdf'];
        if (!empty($link)) {
            $actions[$this->id . '-print'] = array(
                'url'  => \esc_url($link),
                'name' => __('Visualizar boleto', GALAX_IDENTIFIER),

            );
        }
        return $actions;
    }
}
