<?php

/**
 * Plugin Name: Galax Pay WooCommerce
 * Plugin URI:
 * Description: Adiciona o gateway de pagamentos do Galax Pay para o WooCommerce.
 * Version: 4.2
 * Author: Galax Pay
 * Author URI: https://www.galaxpay.com.br
 * Requires at least: 4.4
 * Tested up to: 4.4
 * WC requires at least: 3.0.0
 * WC tested up to: 3.8.1
 *
 * Text Domain: galax-pay-woocommerce
 * Domain Path: /languages/
 *
 * Copyright: © 2020 Galax Pay Pagamentos Eletrônicos LTDA
 * License: GPLv3 or later
 * License URI: http://www.gnu.org/licenses/gpl-3.0.html
 */
if (defined('ABSPATH') == false) {
    die;
}
define('GALAXPAY_DEBUG', true);
define('GALAXPAY_DEEP_LOG',false);
define('GALAX_IDENTIFIER', 'GalaxPayWooCommerce');

require_once dirname(__FILE__) . "/includes/GalaxPayDependencies.php";

if (GalaxPayDependencies::check() == false) {
    return;
}

class GalaxPayWooCommerce
{

    const VERSION = '1.0';
    const VIEWS_DIR = '/templates/';
    const INCLUDES_DIR = '/includes/';
    const WC_API_CALLBACK = 'galax_pay_webhook';

    protected static $instance = null;
    protected $settings = null;
    private $webhook_handler = null;
    private $subscriptionStatus = null;
    private $subscription_status_handler = null;

    public function __construct()
    {
        $this->includes(array(
            'GalaxPayApi.php',
            'GalaxPayCharge.php',
            'GalaxPayGetInfoOrder.php',
            'GalaxPaySubscription.php',
            'GalaxPayBaseGateway.php',
            'GalaxPayBoleto.php',
            'GalaxPayPix.php',
            'GalaxPayCreditCard.php',
            'GalaxPayLogger.php',
            'GalaxPayPayment.php',
            'GalaxPaySettings.php',
            'GalaxActionStatusWebhook.php',
            'GalaxPayWebhooks.php',
            'GalaxPaySubscriptionStatus.php',
            'GalaxPayChargeStatus.php',
            'GalaxPayDefaultStatus.php'
        ));

        $this->settings = new GalaxPaySettings();
        $this->webhook_handler = new GalaxPayWebhooks($this->settings);
        $this->subscriptionStatus = new GalaxPaySubscriptionStatus($this->settings);
        $this->chargeStatus = new GalaxPayChargeStatus($this->settings);
        $this->appendActionsAndFilters();
        $this->appendAdminActions();
        date_default_timezone_set('America/Sao_Paulo');
    }

    public static function getInstance()
    {
        if (self::$instance == null) {
            self::$instance = new self;
        }
        return self::$instance;
    }

    private function appendActionsAndFilters()
    {
        add_action('http_api_curl', [&$this, 'add_support_to_tlsv1_2']);
        add_action('woocommerce_api_' . self::WC_API_CALLBACK, array($this->webhook_handler, 'handle'));
        add_action('woocommerce_add_to_cart_validation', array(&$this, 'validate_add_to_cart'), 1, 3);
        add_filter('plugin_action_links_' . plugin_basename(__FILE__), array(&$this, 'action_links'));
        add_filter('woocommerce_my_account_my_orders_actions', array(&$this, 'user_related_orders_actions'), 100, 2);
        add_filter('woocommerce_subscription_period_interval_strings', array(&$this, 'set_supported_intervals'));
        add_action('woocommerce_customer_save_address', array(&$this, 'updateCustomer'), 1, 2);
        add_action('woocommerce_save_account_details', array(&$this, 'updateCustomerDetails'), 1, 2);
        add_filter('wcs_view_subscription_actions', array(&$this, 'removeMyAccountButtons'), 100, 2);
        add_action( 'woocommerce_admin_order_data_after_order_details', array(&$this, 'addMessageWarning'), 20, 3 );
    }


    private function getOnlyNumbers($doc)
    {
        return preg_replace('/\D/', '', $doc);
    }

    public function updateCustomerDetails($userId)
    {
        $nonce = sanitize_text_field($_POST['save-account-details-nonce']);
        if (wc_notice_count('error') > 0 || empty($nonce)) {
            return;
        }

        $myId = wp_get_current_user()->ID;

        $customer  = array(
            'name' => sanitize_text_field($_POST['account_first_name'] . ' ' . $_POST['account_last_name']),
            'emails' => [sanitize_text_field($_POST['account_email'])]
        );

        $this->settings->api->updateCustomerByMyId($customer, $myId);
    }

    protected function getUserDocumentAndName()
    {
        if ($_POST['billing_persontype'] == '2') {
            $name =  $_POST['billing_company'];
            $cpf_or_cnpj = $_POST['billing_cnpj'];
        } else {
            $name = $_POST['billing_first_name']  . ' ' . $_POST['billing_last_name'];
            $cpf_or_cnpj =  $_POST['billing_cpf'];
        }
        return array('name' => sanitize_text_field($name), 'document' => sanitize_text_field($cpf_or_cnpj));
    }

    public function updateCustomer($user_id)
    {
        $wcsnonce = sanitize_text_field($_POST['_wcsnonce']);
        if (wc_notice_count('error') > 0 || empty($wcsnonce)) {
            return;
        }
        $myId = wp_get_current_user()->ID;
        $userDocumentAndName = $this->getUserDocumentAndName();
        $phones =  [
            $this->getOnlyNumbers(sanitize_text_field($_POST['billing_phone'])),
            $this->getOnlyNumbers(sanitize_text_field($_POST['billing_cellphone']))
        ];
        $customer  = array(
            'name' => $userDocumentAndName['name'],
            'document' => $this->getOnlyNumbers($userDocumentAndName['document']),
            'emails' => [sanitize_text_field($_POST['billing_email'])],
            'phones' => array_filter($phones),
            'Address' => [
                'street' => sanitize_text_field($_POST['billing_address_1']),
                'number' => sanitize_text_field($_POST['billing_number']),
                'complement' => sanitize_text_field($_POST['billing_address_2']),
                'zipCode' => $this->getOnlyNumbers(sanitize_text_field($_POST['billing_postcode'])),
                'neighborhood' => sanitize_text_field($_POST['billing_neighborhood']),
                'city' => sanitize_text_field($_POST['billing_city']),
                'state' => sanitize_text_field($_POST['billing_state']),
            ],
        );

        $this->settings->api->updateCustomerByMyId($customer, $myId);
    }

    function removeMyAccountButtons($actions, $subscription)
    {
        unset($actions['resubscribe']);
        return $actions;
    }

    private function appendAdminActions()
    {
        if (is_admin()) {
            add_action('admin_enqueue_scripts', array(&$this, 'add_admin_scripts'));
        }
    }

    public function set_supported_intervals($current_intervals)
    {
        foreach (range(7, 60) as $new_interval) {
            array_push($current_intervals, $new_interval);
        }
        return $current_intervals;
    }

    public function includes(array $classes)
    {
        foreach ($classes as $class) {
            include_once(dirname(__FILE__) . self::INCLUDES_DIR . $class);
        }
    }

    public static function generate_assets_url($path)
    {
        return plugin_dir_url(__FILE__) . 'assets/' . $path;
    }

    public function action_links($links)
    {
        $links[] = '<a href="admin.php?page=wc-settings&tab=settings_galax_pay">' . __('Configurações', GALAX_IDENTIFIER) . '</a>';
        return $links;
    }

    public function validate_add_to_cart($valid, $product_id, $quantity)
    {
        $cart = $this->settings->woocommerce->cart;
        $cart_items = $cart->get_cart();

        $product = wc_get_product($product_id);

        if (empty($cart_items)) {
            return $valid;
        }
        return $valid;
    }

    public function user_related_orders_actions($actions, $order)
    {
        $filtred_actions = $this->filter_actions($actions, array(
            'pay',
            'cancel',
        ));

        return $filtred_actions;
    }

    private function filter_actions($actions, $filter)
    {
        $filtred_actions = array();
        if (!is_array($actions)) {
            $actions = array();
        }
        $filtred_actions_keys = array_diff(array_keys($actions), $filter);

        foreach ($filtred_actions_keys as $key)
            $filtred_actions[$key] = $actions[$key];

        return $filtred_actions;
    }

    public function add_admin_scripts()
    {
        return $this->settings->add_script('js/simple-subscription-fields.js', array('jquery'));
    }

    public function add_support_to_tlsv1_2($ch)
    {
        if (empty($ch)) {
            return;
        }
        $host_to = parse_url(curl_getinfo($ch, CURLINFO_EFFECTIVE_URL), PHP_URL_HOST);

        if ($host_to !== 'app.galaxpay.com.br') {
            return;
        }
        if (!defined('CURL_SSLVERSION_TLSv1_2')) {
            return;
        }
        curl_setopt($ch, CURLOPT_SSLVERSION, CURL_SSLVERSION_TLSv1_2);
    }
    
    public function addMessageWarning($order )
    {
        echo '<div class="form-field form-field-wide"><h3>OBS: Apenas alterações de status serão enviadas para o Galax Pay. Demais alterações realizadas pelo painel do administrador, como valor, taxas, cupons, não serão refletidas dentro do Galax Pay.</h3></div>';
    }
}

add_action('wp_loaded', array('GalaxPayWooCommerce', 'getInstance'), 0);
