<?php

class GalaxPayApi
{

    public $current_plan;
    public $last_error = '';
    private $galaxId;
    private $galaxHash;
    private $accept_bank_slip;
    private static $paymentMethods = null;

    public function base_path()
    {
        return 'https://api.galaxpay.com.br/v2/';
    }

    public function __construct($galaxId, $galaxHash, $sandbox, $logger = null)
    {
        $this->galaxId = $galaxId;
        $this->galaxHash = $galaxHash;
        $this->logger = $logger;
        $this->sandbox = $sandbox;
    }

    private function build_body($data)
    {
        $body = json_encode($data);
        return $body;
    }

    private function error($error)
    {
        if (!empty($error['message'])) {
            $this->last_error = $error['message'];
            if (!isset($error['details'])) {
                $error['details'] = '';
            }
            $completeMessage = $error['message'] . ' - ' . json_encode($error['details']);
            throw new \Exception($completeMessage);
        }
    }

    private function checkResponse($response, $body)
    {
        $this->last_error = '';
        $code = wp_remote_retrieve_response_code($response);
        if (substr($code, 0, 2) != '20') {
            $this->error($body['error']);
            return false;
        }
        return true;
    }

    public function getIntegrationsActiveInAccount()
    {
        self::$paymentMethods = isset($_SESSION['paymentMethosGalaxPay']) ? $_SESSION['paymentMethosGalaxPay'] : null;
        if (self::$paymentMethods != null) {
            return self::$paymentMethods;
        }
        $res = $this->request('payment-methods', 'GET');
        $allowed = ['pix', 'boleto', 'credit'];
        foreach ($res['PaymentMethods'] as $key => $method) {
            if (!in_array($method['id'], $allowed)) {
                unset($res['PaymentMethods'][$key]);
            }
        }
        sort($res['PaymentMethods']);
        $_SESSION['paymentMethosGalaxPay'] = $res['PaymentMethods'];
        self::$paymentMethods = $res['PaymentMethods'];
        return self::$paymentMethods;
    }

    private function isTokenExpired()
    {
        if (!isset($_SESSION['tokenSavedExpiresInGalaxPay'])) {
            return true;
        }
        if (!isset($_SESSION['tokenSavedCreatedAtGalaxPay'])) {
            return true;
        }
        $seconds = $_SESSION['tokenSavedExpiresInGalaxPay'] - 60; //margem de seguranca de 1 minuto
        $date_now = $_SESSION['tokenSavedCreatedAtGalaxPay'];

        $validUntil = date("Y-m-d H:i:s", (strtotime(date($date_now)) + $seconds));
        $now = date('Y-m-d H:i:s');
        if ($validUntil <= $now) {
            return true;
        }
        return false;
    }

    public function getTokenToUse()
    {
        $tokenSaved = isset($_SESSION['tokenSavedGalaxPay']) ? $_SESSION['tokenSavedGalaxPay'] : null;
        if (!empty($tokenSaved) && !$this->isTokenExpired()) {
            return $tokenSaved;
        }
        $url = $this->base_path() . 'token';
        $data = array(
            'grant_type' => 'authorization_code',
            'scope' => 'payment-methods.read customers.read customers.write plans.read plans.write transactions.read transactions.write webhooks.write cards.read cards.write card-brands.read subscriptions.read subscriptions.write charges.read charges.write boletos.read'
        );
        $body = $this->build_body($data);
        $config = array(
            'headers' => array(
                'Authorization' => 'Basic ' . base64_encode($this->galaxId . ':' . $this->galaxHash),
                'Content-Type' => 'application/json',
            ),
            'timeout' => 60,
            'sslverify' => false,
            'body' => $body,
        );
        $response = wp_remote_post($url, $config);

        if (is_wp_error($response)) {
            return false;
        }

        $response_body = wp_remote_retrieve_body($response);
        if (!$response_body) {
            return false;
        }

        $response_body_array = json_decode($response_body, true);
        $_SESSION['tokenSavedGalaxPay'] = $response_body_array['access_token'];
        $_SESSION['tokenSavedExpiresInGalaxPay'] = $response_body_array['expires_in'];
        $_SESSION['tokenSavedCreatedAtGalaxPay'] = date('Y-m-d H:i:s');
        return $response_body_array['access_token'];
    }

    public function cancelCharge($id, $typeId = 'myId')
    {
        return $this->request('charges/' . $id . '/' . $typeId, 'DELETE');
    }

    public function cancelSubscription($id, $typeId = 'myId')
    {
        return $this->request('subscriptions/' . $id . '/' . $typeId, 'DELETE');
    }

    public function retrySubscription($id, $body)
    {
        return $this->request('subscriptions/' . $id . '/myId', 'PUT', $body);
    }

    public function retryCharge($id, $body)
    {
        return $this->request('charges/' . $id . '/myId', 'PUT', $body);
    }
    
    public function editCharge($id, $body, $typeId = 'galaxPayId')
    {
        return $this->request('charges/' . $id . '/' . $typeId, 'PUT', $body);
    }

    public function setWebhookUrl($url)
    {
        $body = [
            'url' => $url,
            'events' => ["transaction.updateStatus"]
        ];
        return $this->request('webhooks', 'PUT', $body);
    }

    public function getSubscriptionById($id)
    {
        return $this->request('subscriptions?startAt=0&limit=1&myIds=' . $id, 'GET');
    }

    public function getChargeByMyId($id)
    {
        return $this->request('charges?startAt=0&limit=1&myIds=' . $id, 'GET');
    }

    public function createCharge($body)
    {
        return $this->request('charges', 'POST', $body);
    }

    public function createSubscription($body)
    {
        return $this->request('subscriptions', 'POST', $body);
    }

    public function updateCustomerByMyId($body, $myId)
    {
        return $this->request('customers/' . $myId . '/myId', 'PUT', $body);
    }

    private function request($endpoint, $method = 'POST', $data = array(), $debug = false)
    {

        $requestToken = $this->getTokenToUse();
        if ($requestToken === false) {
            return false;
        }
        $url = $this->base_path() . $endpoint;
        $body = '';
        if (count($data) > 0) {
            $body = $this->build_body($data);
        }
        
        $response = wp_remote_post($url, array(
            'headers' => array(
                'Authorization' => 'Bearer ' . $requestToken,
                'Content-Type' => 'application/json',
            ),
            'method' => $method,
            'timeout' => 60,
            'sslverify' => false,
            'body' => $body,
        ));

        if ($debug) {
            var_dump($url, $response);die;
        }
        
        $response_body = wp_remote_retrieve_body($response);

        if (is_wp_error($response)) {

            return false;
        }
        $debug = true;
        if ($debug) {
            $code = wp_remote_retrieve_response_code($response);
            if (substr($code, 0, 2) != '20') {
                if (GALAXPAY_DEBUG) {
                    var_dump($endpoint, $body, $response_body, $requestToken);
                }
            }
        }

        if (!$response_body) {
            return false;
        }

        $response_body_array = json_decode($response_body, true);
        $this->checkResponse($response, $response_body_array);
        return $response_body_array;
    }

    public function get_payment_methods()
    {
        $payment_methods = get_transient('galax_pay_payment_methods');
        $payment_methods = array(
            'credit_card' => false,
            'bank_slip' => false,
            'pix' => false,
        );

        $paymentMethods = $this->getIntegrationsActiveInAccount();

        foreach ($paymentMethods as $method) {
            if ($method['id'] == 'creditcard' || $method['id'] == 'credit') {
                $payment_methods['credit_card'] = true;
            }
            if ($method['id'] == 'boleto') {
                $payment_methods['bank_slip'] = true;
            }
            if ($method['id'] == 'pix') {
                $payment_methods['pix'] = true;
            }
        }
        set_transient('galax_pay_payment_methods', $payment_methods, 1 * HOUR_IN_SECONDS);
        $this->accept_bank_slip = $payment_methods['bank_slip'];
        return $payment_methods;
    }

    public function accept_bank_slip()
    {
        if (null === $this->accept_bank_slip) {
            $this->get_payment_methods();
        }
        return $this->accept_bank_slip;
    }

    public function areCredentialsValid($company)
    {
        $galaxId = $company['id'];
        $galaxHash = $company['galaxHash'];

        if ($galaxHash == $this->galaxHash) {
            return $galaxId == $this->galaxId;
        }
        return false;
    }

    public function putIdOrderAtTransaction($idOrder, $idTransaction)
    {
        $body = [
            'myId' => $idOrder
        ];
        return $this->request('transactions/' . $idTransaction . '/galaxPayId', 'PUT', $body);
    }
   
}
